' SAP ZPACK Packaging Information Extraction Script
' Author: Dario Pascoal
'
' Description: This VBScript automates the extraction of packaging information from SAP's
' ZPACK transaction for tools and equipment. It retrieves Transport Assembly (TA),
' Cleanroom Assembly (CA), and Send Assembly (SA) sizes which are critical for
' proper tool packaging and shipping procedures.
'
' Business Context: Packaging information is essential for DHL field service operations
' to determine correct packaging requirements for tool transportation, storage, and
' shipping. This automation eliminates manual ZPACK lookups and ensures consistent
' packaging data is available in the backlog management system.
'
' Input Parameters:
' - material: The material number to query for packaging information
'
' Output: JSON formatted packaging data containing TA, CA, and SA values
'
' SAP Transaction: ZPACK (Packaging Master Data)
' Integration: Used by backlog.js to populate ZPACK column with packaging details

' Establish SAP GUI connection using standard connection pattern
If Not IsObject(application) Then
   Set SapGuiAuto  = GetObject("SAPGUI")
   Set application = SapGuiAuto.GetScriptingEngine
End If
If Not IsObject(connection) Then
   Set connection = application.Children(0)
End If
If Not IsObject(session) Then
   Set session    = connection.Children(0)
End If

' Connect script event handlers for SAP GUI automation
If IsObject(WScript) Then
   WScript.ConnectObject session,     "on"
   WScript.ConnectObject application, "on"
End If

' Function to safely extract text content from SAP GUI fields with enhanced debugging
' This function implements retry logic to handle timing issues with SAP GUI
' rendering and ensures reliable data extraction even under slow system conditions
Function GetFieldText(fieldId)
    On Error Resume Next
    Dim i, ctrl
    
    ' Debug output for field access attempts
    WScript.Echo "DEBUG: Attempting to access field: " & fieldId
    
    ' Retry mechanism for field access (up to 10 attempts with 300ms delays)
    ' SAP GUI fields may not be immediately available after navigation
    For i = 1 To 10
        Set ctrl = session.findById(fieldId)
        If Err.Number = 0 And Not ctrl Is Nothing Then
            Dim fieldValue
            fieldValue = ctrl.text
            If Err.Number = 0 Then
                WScript.Echo "DEBUG: Field access successful on attempt " & i & ": """ & fieldValue & """"
                GetFieldText = fieldValue
                Exit Function
            Else
                WScript.Echo "DEBUG: Field value read failed on attempt " & i & ": " & Err.Description
            End If
        Else
            WScript.Echo "DEBUG: Field not found on attempt " & i & ": " & Err.Description
        End If
        Err.Clear
        WScript.Sleep 300
    Next
    
    ' Return empty string if field cannot be accessed after all retries
    WScript.Echo "DEBUG: Field access failed after all attempts: " & fieldId
    GetFieldText = ""
    On Error GoTo 0
End Function

' Main ZPACK data extraction procedure
' Accepts material number as command line argument for flexible integration
Dim material
If WScript.Arguments.Count > 0 Then
    material = WScript.Arguments(0)
Else
    WScript.Echo "ERROR: Material number required as argument"
    WScript.Quit 1
End If

' Validate material input to prevent empty queries
If Trim(material) = "" Then
    WScript.Echo "ERROR: Empty material number provided"
    WScript.Quit 1
End If

' Enhanced debugging output for VBScript execution tracing
WScript.Echo "DEBUG: Starting ZPACK extraction for material: " & material
WScript.Echo "DEBUG: SAP connection status check..."

' Verify SAP connection is still active before proceeding
On Error Resume Next
Dim connectionTest
Set connectionTest = session.findById("wnd[0]")
If Err.Number <> 0 Or connectionTest Is Nothing Then
    WScript.Echo "ERROR: SAP connection lost or invalid"
    WScript.Echo "DEBUG: SAP connection error: " & Err.Description
    WScript.Quit 1
End If
On Error GoTo 0
WScript.Echo "DEBUG: SAP connection verified successfully"

' Navigate to ZPACK transaction with enhanced error tracking
WScript.Echo "DEBUG: Navigating to main menu..."
On Error Resume Next
Dim i
For i = 1 To 3
    session.findById("wnd[0]").sendVKey 3  ' Press F3 to go back/exit
    WScript.Sleep 300
Next
On Error GoTo 0

' Maximize window for consistent field identification across different screen resolutions
WScript.Echo "DEBUG: Maximizing SAP window..."
On Error Resume Next
session.findById("wnd[0]").maximize
If Err.Number <> 0 Then
    WScript.Echo "DEBUG: Window maximize failed: " & Err.Description
End If
On Error GoTo 0

' Navigate to ZPACK transaction with error handling
WScript.Echo "DEBUG: Entering ZPACK transaction..."
On Error Resume Next
session.findById("wnd[0]/tbar[0]/okcd").text = "zpack"
If Err.Number <> 0 Then
    WScript.Echo "ERROR: Cannot access transaction command field"
    WScript.Echo "DEBUG: Transaction field error: " & Err.Description
    WScript.Quit 1
End If
session.findById("wnd[0]").sendVKey 0
If Err.Number <> 0 Then
    WScript.Echo "ERROR: Cannot execute ZPACK transaction"
    WScript.Echo "DEBUG: Transaction execution error: " & Err.Description
    WScript.Quit 1
End If
On Error GoTo 0
WScript.Sleep 1000
WScript.Echo "DEBUG: ZPACK transaction opened successfully"

' Input material number and execute query with enhanced debugging
WScript.Echo "DEBUG: Setting material number: " & material
On Error Resume Next
Dim materialField
Set materialField = session.findById("wnd[0]/usr/ctxtS_MATNR-LOW")
If Err.Number <> 0 Or materialField Is Nothing Then
    WScript.Echo "ERROR: Cannot access material input field"
    WScript.Echo "DEBUG: Material field error: " & Err.Description
    WScript.Echo "DEBUG: Error number: " & Err.Number
    WScript.Quit 1
End If
materialField.text = material
If Err.Number <> 0 Then
    WScript.Echo "ERROR: Cannot set material number"
    WScript.Echo "DEBUG: Material set error: " & Err.Description
    WScript.Quit 1
End If
WScript.Echo "DEBUG: Material number set successfully"

' Execute the search with error handling
WScript.Echo "DEBUG: Executing ZPACK search..."
session.findById("wnd[0]").sendVKey 8
If Err.Number <> 0 Then
    WScript.Echo "ERROR: Cannot execute ZPACK search"
    WScript.Echo "DEBUG: Search execution error: " & Err.Description
    WScript.Quit 1
End If
On Error GoTo 0
WScript.Sleep 1500
WScript.Echo "DEBUG: ZPACK search completed"

' Check for error messages or empty results
WScript.Echo "DEBUG: Checking for error messages..."
On Error Resume Next
Dim statusBar
Set statusBar = session.findById("wnd[0]/sbar")
If Not statusBar Is Nothing Then
    Dim statusMessage
    statusMessage = statusBar.text
    If statusMessage <> "" Then
        WScript.Echo "DEBUG: SAP status message: " & statusMessage
        If InStr(UCase(statusMessage), "ERROR") > 0 Or InStr(UCase(statusMessage), "NOT FOUND") > 0 Then
            WScript.Echo "ERROR: Material not found or invalid: " & statusMessage
            WScript.Quit 1
        End If
    End If
End If
On Error GoTo 0

' Wait for transaction to process and navigate to packaging tab with enhanced error handling
WScript.Echo "DEBUG: Navigating to packaging tab..."
On Error Resume Next
Dim packagingTab
Set packagingTab = session.findById("wnd[0]/usr/tabsTAB_9100/tabpTAB_9100_FC3")
If Err.Number <> 0 Or packagingTab Is Nothing Then
    WScript.Echo "ERROR: Cannot access packaging tab - material may not exist or no packaging data"
    WScript.Echo "DEBUG: Packaging tab error: " & Err.Description
    WScript.Echo "DEBUG: Error number: " & Err.Number
    WScript.Quit 1
End If
packagingTab.select
If Err.Number <> 0 Then
    WScript.Echo "ERROR: Cannot select packaging tab"
    WScript.Echo "DEBUG: Tab selection error: " & Err.Description
    WScript.Quit 1
End If
On Error GoTo 0
WScript.Sleep 1000
WScript.Echo "DEBUG: Packaging tab selected successfully"

' Extract packaging assembly information using direct field access with enhanced debugging
WScript.Echo "DEBUG: Extracting packaging data from fields..."
Dim transportAssy, cleanAssy, sendAssy

' Attempt to read packaging data from standard field locations with debugging
WScript.Echo "DEBUG: Reading Transport Assembly field..."
transportAssy = GetFieldText("wnd[0]/usr/tabsTAB_9100/tabpTAB_9100_FC3/ssubTAB_9100_SCA:ZDGP_MATERIAL_OVERVIEW:9103/txtWA_OUTPUT_9100-TXTPACK_TRANSP_SIZE")
WScript.Echo "DEBUG: Transport Assembly: """ & transportAssy & """"

WScript.Echo "DEBUG: Reading Clean Assembly field..."
cleanAssy = GetFieldText("wnd[0]/usr/tabsTAB_9100/tabpTAB_9100_FC3/ssubTAB_9100_SCA:ZDGP_MATERIAL_OVERVIEW:9103/txtWA_OUTPUT_9100-PACK_CRBOXT")
WScript.Echo "DEBUG: Clean Assembly: """ & cleanAssy & """"

WScript.Echo "DEBUG: Reading Send Assembly field..."
sendAssy = GetFieldText("wnd[0]/usr/tabsTAB_9100/tabpTAB_9100_FC3/ssubTAB_9100_SCA:ZDGP_MATERIAL_OVERVIEW:9103/txtWA_OUTPUT_9100-PACK_SENDBOXT")
WScript.Echo "DEBUG: Send Assembly: """ & sendAssy & """"

' Fallback data extraction using table scanning method with enhanced debugging
WScript.Echo "DEBUG: Checking if fallback table scan is needed..."
If transportAssy = "" Or cleanAssy = "" Or sendAssy = "" Then
    WScript.Echo "DEBUG: Direct field access incomplete, attempting table scan fallback..."
    On Error Resume Next
    Set table = session.findById("wnd[0]/usr/tabsTAB_9100/tabpTAB_9100_FC3/ssubTAB_9100_SCA:ZDGP_MATERIAL_OVERVIEW:9103/cntlCONTROL_9103_3/shellcont/shell")
    
    If Err.Number = 0 And Not table Is Nothing Then
        WScript.Echo "DEBUG: Table found, scanning for packaging entries..."
        Dim rowCount
        rowCount = table.RowCount
        WScript.Echo "DEBUG: Table has " & rowCount & " rows"
        
        ' Scan table rows for packaging assembly descriptions
        Dim rowIndex, desc
        For rowIndex = 0 To rowCount - 1
            On Error Resume Next
            desc = table.GetCellValue(rowIndex, "MAKTX")
            On Error GoTo 0
            
            If desc <> "" Then
                WScript.Echo "DEBUG: Row " & rowIndex & " description: """ & desc & """"
                ' Enhanced pattern matching for assembly types using multiple patterns
                Dim upperDesc
                upperDesc = UCase(desc)
                
                ' Transport Assembly patterns
                If transportAssy = "" Then
                    If InStr(upperDesc, " TA ") > 0 Or _
                       InStr(upperDesc, "TA FOR") > 0 Or _
                       InStr(upperDesc, "TRANSPORT") > 0 Or _
                       InStr(upperDesc, "TA STD") > 0 Or _
                       InStr(upperDesc, " TA") > 0 Or _
                       InStr(upperDesc, "TOOL TA") > 0 Then
                        transportAssy = desc
                        WScript.Echo "DEBUG: Found TA in table: """ & transportAssy & """"
                    End If
                End If
                
                ' Clean Assembly patterns  
                If cleanAssy = "" Then
                    If InStr(upperDesc, " CA ") > 0 Or _
                       InStr(upperDesc, "CA FOR") > 0 Or _
                       InStr(upperDesc, "CLEAN") > 0 Or _
                       InStr(upperDesc, "CA STD") > 0 Or _
                       InStr(upperDesc, "CA ESD") > 0 Or _
                       InStr(upperDesc, " CA") > 0 Or _
                       InStr(upperDesc, "TOOL CA") > 0 Then
                        cleanAssy = desc
                        WScript.Echo "DEBUG: Found CA in table: """ & cleanAssy & """"
                    End If
                End If
                
                ' Send Assembly patterns
                If sendAssy = "" Then
                    If InStr(upperDesc, " SA ") > 0 Or _
                       InStr(upperDesc, "SA FOR") > 0 Or _
                       InStr(upperDesc, "SEND") > 0 Or _
                       InStr(upperDesc, "SA STD") > 0 Or _
                       InStr(upperDesc, " SA") > 0 Or _
                       InStr(upperDesc, "TOOL SA") > 0 Or _
                       InStr(upperDesc, "STD CARD BOX") > 0 Then
                        sendAssy = desc
                        WScript.Echo "DEBUG: Found SA in table: """ & sendAssy & """"
                    End If
                End If
                
                ' Exit loop early if all assembly types are found
                If transportAssy <> "" And cleanAssy <> "" And sendAssy <> "" Then
                    WScript.Echo "DEBUG: All packaging data found in table"
                    Exit For
                End If
            End If
        Next
    Else
        WScript.Echo "DEBUG: Table not found or inaccessible: " & Err.Description
    End If
    On Error GoTo 0
Else
    WScript.Echo "DEBUG: Direct field access successful, skipping table scan"
End If

WScript.Echo "DEBUG: Final packaging data:"
WScript.Echo "DEBUG: - Transport Assembly: """ & transportAssy & """"
WScript.Echo "DEBUG: - Clean Assembly: """ & cleanAssy & """"
WScript.Echo "DEBUG: - Send Assembly: """ & sendAssy & """"

' Output structured JSON data for JavaScript consumption with completion status
WScript.Echo "DEBUG: Generating JSON output for material: " & material
Dim jsonOutput
jsonOutput = "{" & _
    """material"": """ & material & """," & _
    """transportAssy"": """ & Replace(transportAssy, """", "\""") & """," & _
    """cleanAssy"": """ & Replace(cleanAssy, """", "\""") & """," & _
    """sendAssy"": """ & Replace(sendAssy, """", "\""") & """," & _
    """status"": ""success""" & _
    "}"

WScript.Echo "DEBUG: JSON output generated successfully"
WScript.Echo jsonOutput

' Navigate back to SAP main menu for next script execution with enhanced error handling
WScript.Echo "DEBUG: Navigating back to main menu..."
On Error Resume Next

' First F3 - Exit current screen
session.findById("wnd[0]").sendVKey 3
If Err.Number <> 0 Then
    WScript.Echo "DEBUG: First F3 navigation warning: " & Err.Description
End If
WScript.Sleep 500

' Check for warning dialog that requires "Leave" button press
Dim leaveDialog
Set leaveDialog = session.findById("wnd[1]")
If Err.Number = 0 And Not leaveDialog Is Nothing Then
    WScript.Echo "DEBUG: Warning dialog detected, pressing Leave button..."
    Dim leaveButton
    Set leaveButton = session.findById("wnd[1]/tbar[0]/btn[12]")
    If Err.Number = 0 And Not leaveButton Is Nothing Then
        WScript.Echo "DEBUG: Pressing Leave button to dismiss warning dialog..."
        leaveButton.press
        WScript.Sleep 800
        WScript.Echo "DEBUG: Leave button pressed successfully - this counts as going back one window"
        ' Since pressing "Leave" goes back one window, we only need 2 more F3 presses instead of 3
        session.findById("wnd[0]").sendVKey 3  ' Second F3 - Exit transaction
        If Err.Number <> 0 Then
            WScript.Echo "DEBUG: Second F3 navigation warning: " & Err.Description
        End If
        WScript.Sleep 500
        session.findById("wnd[0]").sendVKey 3  ' Third F3 - Return to main menu
        If Err.Number <> 0 Then
            WScript.Echo "DEBUG: Third F3 navigation warning: " & Err.Description
        End If
        WScript.Sleep 500
    Else
        WScript.Echo "DEBUG: Leave button not found, proceeding with normal navigation..."
        ' Normal flow - 2 more F3 presses
        session.findById("wnd[0]").sendVKey 3  ' Second F3 - Exit transaction
        If Err.Number <> 0 Then
            WScript.Echo "DEBUG: Second F3 navigation warning: " & Err.Description
        End If
        WScript.Sleep 500  
        session.findById("wnd[0]").sendVKey 3  ' Third F3 - Return to main menu
        If Err.Number <> 0 Then
            WScript.Echo "DEBUG: Third F3 navigation warning: " & Err.Description
        End If
        WScript.Sleep 500
    End If
Else
    WScript.Echo "DEBUG: No warning dialog detected, proceeding with normal navigation..."
    ' Normal flow - 2 more F3 presses
    session.findById("wnd[0]").sendVKey 3  ' Second F3 - Exit transaction
    If Err.Number <> 0 Then
        WScript.Echo "DEBUG: Second F3 navigation warning: " & Err.Description
    End If
    WScript.Sleep 500  
    session.findById("wnd[0]").sendVKey 3  ' Third F3 - Return to main menu
    If Err.Number <> 0 Then
        WScript.Echo "DEBUG: Third F3 navigation warning: " & Err.Description
    End If
    WScript.Sleep 500
End If

On Error GoTo 0
WScript.Echo "DEBUG: ZPACK script completed successfully for material: " & material
